<?php
/**
 * 2022 4webs Hipercalzado
 *
 * DEVELOPED by 4webs.es Prestashop Superhero Partner
 *
 * @author    4webs
 * @copyright 4webs 2022
 * @license   4webs
 * @category administration
 */

class HipercalzadoOrder extends ObjectModel
{
    public $id_hc_order;
    public $hc_id_order;
    public $st_id_order;
    public $id_cart;
    public $id_customer;
    public $shipping_company;
    public $can_cancel;
    public $date_add;

    public static $definition = array(
        'table' => 'hipercalzado_order',
        'primary' => 'id_hipercalzado_order',
        'fields' => array(
            'hc_id_order' => array('type' => self::TYPE_INT),
            'st_id_order' => array('type' => self::TYPE_INT),
            'id_cart' => array('type' => self::TYPE_INT),
            'id_customer' => array('type' => self::TYPE_INT),
            'shipping_company' => array('type' => self::TYPE_STRING),
            'can_cancel' => array('type' => self::TYPE_BOOL),
            'date_add' => array('type' => self::TYPE_DATE)
        ),
    );

    /**
     * Deletes current Hipercalzado order from the database.
     *
     * @return bool True if delete was successful
     *
     * @throws PrestaShopException
     */
    public function delete()
    {
        if (!Db::getInstance()->execute('DELETE FROM `' . _DB_PREFIX_ . 'hipercalzado_order_product` 
                                            WHERE `id_hipercalzado_order` = ' . (int) $this->id)
        ) {
            return false;
        }

        return parent::delete();
    }

    /**
     * Get Hipercalzado order by reference
     *
     * @param $hc_id_order
     * @return false|HipercalzadoOrder
     * @throws PrestaShopDatabaseException
     * @throws PrestaShopException
     */
    public static function getHipercalzadoOrderByHCOrderReference($hc_id_order)
    {
        $id = Db::getInstance()->getValue('SELECT id_hipercalzado_order FROM ' . _DB_PREFIX_ . 'hipercalzado_order WHERE hc_id_order = ' . (int) $hc_id_order);

        if (!$id) {
            return false;
        }

        return new HipercalzadoOrder($id);
    }

    /**
     * Verify if exists Hipercalzado order by reference
     *
     * @param $hc_id_order
     * @return false|HipercalzadoOrder
     * @throws PrestaShopDatabaseException
     * @throws PrestaShopException
     */
    public static function hipercalzadoOrderExists($hc_id_order)
    {
        $query = 'SELECT COUNT(*) FROM ' . _DB_PREFIX_ . 'hipercalzado_order WHERE hc_id_order = ' . (int) $hc_id_order;
        $exists = Db::getInstance()->getValue($query);

        return $exists > 0;
    }

    
    /**
     * Get Hipercalzado order by reference
     *
     * @param $hc_id_order
     * @return false|HipercalzadoOrder
     * @throws PrestaShopDatabaseException
     * @throws PrestaShopException
     */
    public static function getHipercalzadoOrderByIdOrder($st_id_order)
    {
        $id = Db::getInstance()->getValue('SELECT id_hipercalzado_order FROM ' . _DB_PREFIX_ . 'hipercalzado_order WHERE st_id_order = ' . (int) $st_id_order);

        if (!$id) {
            return false;
        }

        return new HipercalzadoOrder($id);
    }
    

    public function hipercalzadoCustomerExists($customer_email)
    {
        if (empty($this->id_customer)) {
            return false;
        }


        $id_customer = Db::getInstance()->getValue(
            'SELECT id_customer FROM ' . _DB_PREFIX_ . 'customer WHERE id_customer = ' . (int) $this->id_customer . ' AND email LIKE \'' . pSQL($customer_email) . '\' AND deleted = 0 AND is_guest = 0'
        );

        if (!$id_customer) {
            return false;
        } else {
            return true;
        }
    }

    public function getHipercalzadoOrder()
    {
        // Existen los ids de pedido y carro
        if (empty($this->st_id_order) || empty($this->id_cart)) {
            return false;
        }

        // El carro existe
        if (!$cart = new Cart($this->id_cart)) {
            return false;
        }

        // El pedido forma parte del carrito
        if (!$cart->orderExists()) {
            return false;
        }

        $id_order_exists = Db::getInstance()->getValue(
            'SELECT id_order FROM ' . _DB_PREFIX_ . 'orders WHERE id_order = ' . (int) $this->st_id_order
        );

        if (!$id_order_exists) {
            return false;
        } else {
            return new Order($this->st_id_order);
        }
    }

    public function createHipercalzadoCustomer($customer_email, $customer_firstname, $customer_lastname)
    {

        // No existe, creamos el cliente
        $PS_customer = new Customer();
        $PS_customer->email = pSQL($customer_email);
        $PS_customer->firstname = pSQL($customer_firstname);
        $PS_customer->lastname = pSQL($customer_lastname);

        // Generamos una contrasenya aleatoria
        //$PS_customer->passwd = Tools::passwdGen(12);
        $PS_customer->passwd = Tools::hash('LapassWord1234es_igual_para_todos_los_clientes_de_Hipercalzado');

        $PS_customer->save();

        if (!$PS_customer) {
            return false;
        }

        // Guardamos el id
        $this->id_customer = (int) $PS_customer->id;
        $this->save();

        return $PS_customer;
    }


    public function createHipercalzadoCart($id_shipping_address, $id_billing_address, $shopDefaultLangId, $id_currency, $id_carrier)
    {

        if (empty($this->id_customer)) {
            return false;
        }

        if($id_currency == '' || is_null($id_currency) || $id_currency == 0) {
            $id_currency = 1;
        }

        $PS_cart = new Cart();
        $PS_cart->id_customer           = (int) $this->id_customer;
        $PS_cart->id_address_delivery   = (int) $id_shipping_address;
        $PS_cart->id_address_invoice    = (int) $id_billing_address;
        $PS_cart->id_lang               = (int) $shopDefaultLangId;
        $PS_cart->id_currency           = (int) $id_currency;
        $PS_cart->id_carrier            = (int) $id_carrier;
        $PS_cart->recyclable            = 0;
        $PS_cart->gift                  = 0;
        $PS_cart->save();

        if (!$PS_cart) {
            return false;
        }

        // Guardamos el id
        $this->id_cart = $PS_cart->id;
        $this->save();

        return $PS_cart;
    }
    

    /*
     * Crea un pedido nuevo con los datos del carrito
     */
    public function createHipercalzadoOrder($module, $context, $WS_order, $order_state, $id_employee, $parent_id, $sku_id)
    {
        if (self::hipercalzadoOrderExists($WS_order->hc_id_order)) {
            return false;
        }    

        // El carro existe
        if (!$cart = new Cart($this->id_cart)) {
            return false;
        }

        $processed_order_lines = self::buildOrderDetailArray($module, $WS_order->order_lines, $cart->id_lang, $parent_id, $sku_id, $cart->id_address_delivery);

        if(is_null($cart->id_currency) || $cart->id_currency == 0 || $cart->id_currency == '') {
            $cart->id_currency = 1;
        }

        $currencyArr = self::getCurrency($cart->id_currency);

        // Creamos el pedido
        $PS_order = new Order();
        $PS_order->id_cart              = (int) $cart->id;
        $PS_order->id_customer          = (int) $cart->id_customer;
        $PS_order->id_address_delivery  = (int) $cart->id_address_delivery;
        $PS_order->id_address_invoice   = (int) $cart->id_address_invoice;
        $PS_order->id_currency          = (int) $cart->id_currency;
        $PS_order->id_lang              = (int) $cart->id_lang;
        $PS_order->reference            = Order::generateReference();
        $PS_order->id_shop              = (int) $context->shop->id;
        $PS_order->id_shop_group        = (int) $context->shop->id_shop_group;
        $PS_order->conversion_rate      = $currencyArr['conversion_rate'] != null ? round($currencyArr['conversion_rate'], 6) : 1;
        $PS_order->id_carrier           = (int) $cart->id_carrier;
        $PS_order->payment              = $WS_order->channel->label == 'Aqui Mejor' ? 'Aqui Mejor' : 'Hipercalzado';
        $PS_order->module               = 'hipercalzado';
        $PS_order->secure_key           = md5('Hipercalzado_pos_sale');
        $PS_order->valid                = 1;
        $PS_order->recyclable           = 0;

        //Totals
        $PS_order->round_mode = Configuration::get(
            'PS_PRICE_ROUND_MODE',
            null,
            (int)$context->shop->id_shop_group,
            (int)$context->shop->id
        );
        $PS_order->round_type = Configuration::get(
            'PS_ROUND_TYPE',
            null,
            (int)$context->shop->id_shop_group,
            (int)$context->shop->id
        );

        // dump($WS_order);
        

        $PS_order->total_paid               = round($WS_order->total_price, 6);
        $PS_order->total_paid_tax_incl      = round($WS_order->total_price, 6);
        $PS_order->total_paid_tax_excl      = round($processed_order_lines['total_paid_tax_excl'], 6);
        $PS_order->total_paid_real          = 0;
        $PS_order->total_products           = round($processed_order_lines['total_products'], 6);
        $PS_order->total_products_wt        = round($processed_order_lines['total_products_wt'], 6);
        
        if ($processed_order_lines['total_shipping_price'] > 0) {
            $PS_order->total_shipping_tax_excl  = round($processed_order_lines['total_shipping_price'], 6);
            $PS_order->total_shipping_tax_incl  = round($processed_order_lines['total_shipping_price_wt'], 6);
        } else {
            $PS_order->total_shipping_tax_excl  = round($WS_order->shipping_price, 6);
            $PS_order->total_shipping_tax_incl  = round($WS_order->shipping_price, 6);
        }

        $PS_order->total_shipping = round($WS_order->shipping_price, 6);

        //dates
        $PS_order->delivery_date = pSQL($WS_order->delivery_date->latest);

        // dump($cart);
        // die(dump($PS_order));
        
        //Add the order only if there's at least one line
        if (!empty($processed_order_lines['lines']) && is_array($processed_order_lines['lines'])) {
            foreach ($processed_order_lines['lines'] as $line) {
                //$line['rawItem']->product_ean = '888757710850';
                //$line['rawItem']->product_ean = '8432319019916';
                // Actualizamos carro // customization are deleted in deleteCustomization
                // Si es producto fake, lo anyadimos al carrito para que quede registrado
                    HipercalzadoUtils::addProductToCart($cart, $line);
                   // $cart->update();
            }

            //Add the order
            if ($PS_order->add()) {
                //Then, add the details
                self::writeOrderDetail($PS_order, $processed_order_lines, $this->id);

                //Register order Carrier
                self::registerOrderCarrier($PS_order, $WS_order->shipping_tracking);

                //Register the payment
                $PS_order->addOrderPayment($PS_order->total_paid, Tools::ucfirst("Hipercalzado"));
                
                //Register status change
                self::registerOrderStatus($PS_order, $order_state, $id_employee);

                // Generate order invoice with correct shipping taxes
                // self::generateInvoice($PS_order);

                // Añadir el comentario al pedido
                $comment = 'ID del pedido en Hipercalzado: ' . $WS_order->order_id . ' - referencia: ' . $WS_order->references->order_reference_for_seller;

                // Crear un nuevo CustomerThread
                $customer_thread = new CustomerThread();
                $customer_thread->id_shop = $PS_order->id_shop;
                $customer_thread->id_lang = $PS_order->id_lang;
                $customer_thread->id_contact = 0; // Puedes establecer el ID de contacto si es necesario
                $customer_thread->id_customer = $PS_order->id_customer;
                $customer_thread->id_order = $PS_order->id;
                $customer_thread->id_product = 0; // Puedes establecer el ID del producto si es necesario
                $customer_thread->status = 'closed'; // Estado cerrado
                $customer_thread->email = $PS_order->getCustomer()->email;
                $customer_thread->token = Tools::passwdGen(12);
                $customer_thread->add();

                // Crear un nuevo CustomerMessage
                $customer_message = new CustomerMessage();
                $customer_message->id_customer_thread = $customer_thread->id;
                $customer_message->id_employee = 1; // ID del empleado que añade el mensaje
                $customer_message->message = $comment;
                $customer_message->private = 1; // 1 para mensaje privado, 0 para mensaje público
                $customer_message->add();

                if ($customer_message->id) {
                    echo 'El mensaje del cliente se ha insertado correctamente.';
                } else {
                    echo 'Error al insertar el mensaje del cliente.';
                    // Puedes obtener más detalles del error si es necesario
                    $error = Db::getInstance()->getMsgError();
                    echo 'Detalles del error: ' . $error;
                }
            }
        }

        if (!$PS_order) {
            return false;
        }

        // Guardamos el id
        $this->st_id_order = (int) $PS_order->id;
        $this->save();

        return $PS_order;
    }

    /**
     * @param $items - The items array
     * @param $langId - The language id (mandatory for some texts)
     *
     * @return array - lines array
     * @todo - reduction
     * Build's a prestashop order lines array
     *
     */
    private static function buildOrderDetailArray($module, $items, $id_lang, $parent_id, $sku_id, $id_address)
    {
        $lines = [];
        $total_paid_tax_excl = 0.00;
        $total_products = 0.00;
        $total_products_wt = 0.00;
        $total_shipping_price = 0.00;
        $total_shipping_price_wt = 0.00;
        $total_paid_without_discount = 0.00;

        // $fake_product_id = (int) HipercalzadoUtils::HIPERCALZADO_ID_NON_EXISTENT_PRODUCT;

        //Iterate the rows
        foreach ($items as $item) {
            $line = null;

            $idSearchPC = $sku_id == 'ean13' ? $item->product_ean : $item->product_sku;
            
            //It's a custom product? or a prestahsop one
            $create_fake_product = false;
            $PS_product = HipercalzadoUtils::getProductByIdentifier($item->product_sku, $parent_id);
            if (empty($PS_product)) {
                // Probaby a combination
                $PS_combination = HipercalzadoUtils::getProductCombinationByIdentifier($idSearchPC, $sku_id);

                if (empty($PS_combination)) {
                    $create_fake_product = true;
                } else {
                    $PS_product = new Product($PS_combination->id_product);
                }
            }

            //Not all products has taxes, this one has?
            $taxID = 0;
            if ($item->taxes != null) {
                if (is_object($item->taxes)) {
                    $taxID = HipercalzadoUtils::getTaxRuleFromRateAndIDAdress($item->taxes->rate, $id_address, (int)Shop::getContextShopID());  
                }
            }
            $unit_price_tax_incl = $item->price_unit;
            $unit_price_tax_excl = ($taxID == 0 && empty($item->taxes->amount)) ? $item->price_unit : round($item->price_unit - $item->taxes->amount, 6);

            // $price_reduction_amount = round($item->price_unit - $item->price, 4);

            $total_price_tax_incl = round($unit_price_tax_incl * $item->quantity, 6);
            $total_price_tax_excl = round($unit_price_tax_excl * $item->quantity, 6);

            $shipping_price_tax_excl    = ($taxID == 0) ? $item->shipping_price : round($item->shipping_price/(1+($item->taxes->rate/100)), 6);
            $shipping_price_tax_incl    = $item->shipping_price;

            // $product_name = $item->product_title;
            //$id_product = 0;
            $id_product_attribute = 0;

            if ($create_fake_product) {
                $PS_product = HipercalzadoUtils::createFakeProduct($module, $item->product_sku);

                /*
                $fake_product_name = HipercalzadoUtils::createMultiLangField('Hipercalzado product not found in Prestashop');
                // Si existe idioma esp, ponemos el nombre en esp
                if($espLangId = Language::getIdByIso('es', true)) {
                    $fake_product_name[$espLangId] = 'Producto de Hipercalzado no encontrado en Prestashop';
                }
                $fake_product_id++;
                */

                //Custom
                $line = [
                    'id_warehouse' => 0,
                    'id_shop' => (int)Shop::getContextShopID(),
                    'product_id' => $PS_product->id,
                    // 'product_id' => $fake_product_id,
                    'product_attribute_id' => 0,
                    'product_name' => $PS_product->name[$id_lang] . ' ' . $module->configuration->parent_id . ': ' . $item->product_sku,
                    // 'product_name' => $fake_product_name[$id_lang],
                    'product_quantity' => $item->quantity,
                    'product_quantity_in_stock' => $item->quantity,
                    'product_quantity_refunded' => 0,
                    'product_quantity_return' => 0,
                    'product_quantity_reinjected' => 0,
                    'product_price' => $unit_price_tax_incl,
                    'reduction_percent' => 0,
                    'reduction_amount' => 0,
                    'reduction_amount_tax_incl' => 0,
                    'group_reduction' => 0,
                    'product_quantity_discount' => 0,
                    'product_ean13' => '',
                    'product_reference' => ($parent_id == 'reference') ? $item->product_sku : '',
                    'id_tax_rules_group' => $taxID,
                    'shipping_price_tax_excl' => $shipping_price_tax_excl,
                    'shipping_price_tax_incl' => $shipping_price_tax_incl,
                    'total_price_tax_incl' => $total_price_tax_incl,
                    'total_price_tax_excl' => $total_price_tax_excl,
                    'unit_price_tax_incl' => $unit_price_tax_incl,
                    'unit_price_tax_excl' => $unit_price_tax_excl,
                    'purchase_supplier_price' => 0.0,
                    'original_product_price' => $unit_price_tax_excl,
                    'original_wholesale_price' => 0.0,
                    'is_fake_product' => true,
                    'id_specific_price' => 0,
                    'rawItem' => $item
                ];

            } else {
                $reference = $PS_product->reference;
                if (!empty($PS_combination)) {
                    $reference = $PS_combination->reference;
                    $id_product_attribute = $PS_combination->id;
                }

                //Prestashop item
                $line = [
                    'id_warehouse' => 0,
                    'id_shop' => (int)Shop::getContextShopID(),
                    'product_id' => $PS_product->id,
                    'product_attribute_id' => $id_product_attribute,
                    'product_name' => Product::getProductName($PS_product->id, $id_product_attribute ,$id_lang),
                    'product_quantity' => $item->quantity,
                    'product_quantity_in_stock' => StockAvailable::getQuantityAvailableByProduct($PS_product->id, $id_product_attribute, (int)Shop::getContextShopID()),
                    'product_quantity_refunded' => 0,
                    'product_quantity_return' => 0,
                    'product_quantity_reinjected' => 0,
                    'product_price' => $unit_price_tax_excl,
                    'reduction_percent' => 0,
                    'reduction_amount' => 0,
                    'reduction_amount_tax_incl' => 0,
                    'group_reduction' => 0,
                    'product_quantity_discount' => 0,
                    'product_ean13' => ($PS_combination != null ? $PS_combination->ean13 : $PS_product->ean13),
                    'product_reference' => $reference,
                    'id_tax_rules_group' => $taxID,
                    'shipping_price_tax_excl' => $shipping_price_tax_excl,
                    'shipping_price_tax_incl' => $shipping_price_tax_incl,
                    'total_price_tax_incl' => $total_price_tax_incl,
                    'total_price_tax_excl' => $total_price_tax_excl,
                    'unit_price_tax_incl' => $unit_price_tax_incl,
                    'unit_price_tax_excl' => $unit_price_tax_excl,
                    'purchase_supplier_price' => round($PS_product->wholesale_price, 4),
                    'original_product_price' => $unit_price_tax_excl,
                    'original_wholesale_price' => round($PS_product->wholesale_price, 4),
                    'is_fake_product' => false,
                    'id_specific_price' => 0,
                    'rawItem' => $item
                ];
            }

            $lines[] = $line;

            //Increase sums
            $total_paid_without_discount    += $total_price_tax_incl;
            $total_paid_tax_excl            += $total_price_tax_excl;
            $total_products                 += $total_price_tax_excl;
            $total_products_wt              += $total_price_tax_incl;
            $total_shipping_price           += $shipping_price_tax_excl;
            $total_shipping_price_wt        += $shipping_price_tax_incl;
        }

        return [
            'lines' => $lines,
            'total_shipping_price' => round($total_shipping_price, 6),
            'total_shipping_price_wt' => round($total_shipping_price_wt, 6),
            'total_paid_without_discount' => round($total_paid_without_discount, 6),
            'total_paid_tax_excl' => round(($total_paid_tax_excl + $total_shipping_price), 6),
            'total_products' => round($total_products, 6),
            'total_products_wt' => round($total_products_wt, 6)
        ];
    }

    private static function writeOrderDetail($order, $lines, $id_hipercalzado_order)
    {
        $taxesArray = [];
        
        //Iterate and write
        foreach ($lines['lines'] as $line) {
            $orderDetail = new OrderDetail();
            $orderDetail->id_warehouse                      = 0;
            $orderDetail->id_shop                           = (int)$order->id_shop;
            $orderDetail->product_id                        = (int)$line['product_id'];
            $orderDetail->product_attribute_id              = (int)$line['product_attribute_id'];
            $orderDetail->product_name                      = pSQL($line['product_name']);
            $orderDetail->product_quantity                  = (int)$line['product_quantity'];
            $orderDetail->product_quantity_in_stock         = (int)$line['product_quantity_in_stock'];
            $orderDetail->product_quantity_refunded         = (int)$line['product_quantity_refunded'];
            $orderDetail->product_quantity_return           = (int)$line['product_quantity_return'];
            $orderDetail->product_quantity_reinjected       = (int)$line['product_quantity_reinjected'];
            $orderDetail->product_price                     = round($line['product_price'], 6);
            $orderDetail->reduction_percent                 = round($line['reduction_percent'], 2);
            $orderDetail->reduction_amount                  = round($line['reduction_amount'], 6);
            $orderDetail->reduction_amount_tax_incl         = round($line['reduction_amount_tax_incl'], 6);
            $orderDetail->group_reduction                   = round($line['group_reduction'], 2);
            $orderDetail->product_quantity_discount         = round($line['product_quantity_discount'], 6);
            $orderDetail->product_ean13                     = pSQL($line['product_ean13']);
            $orderDetail->product_reference                 = pSQL($line['product_reference']);
            $orderDetail->id_tax_rules_group                = (int)$line['id_tax_rules_group'];
            $orderDetail->total_price_tax_incl              = round($line['total_price_tax_incl'], 6);
            $orderDetail->total_price_tax_excl              = round($line['total_price_tax_excl'], 6);
            $orderDetail->unit_price_tax_incl               = round($line['unit_price_tax_incl'], 6);
            $orderDetail->unit_price_tax_excl               = round($line['unit_price_tax_excl'], 6);
            $orderDetail->purchase_supplier_price           = round($line['purchase_supplier_price'], 6);
            $orderDetail->original_product_price            = round($line['original_product_price'], 6);
            $orderDetail->original_wholesale_price          = round($line['original_wholesale_price'], 6);
            $orderDetail->total_shipping_price_tax_excl     = round($line['shipping_price_tax_excl'], 6);
            $orderDetail->total_shipping_price_tax_incl     = round($line['shipping_price_tax_incl'], 6);
            $orderDetail->id_order                          = (int)$order->id;


            //The line has been saved correctly?
            if ($orderDetail->add()) {
                //Decrement the stock, only if product exists in the system
                if (!$line['is_fake_product']) {
                    HipercalzadoUtils::changeProductStock(
                        $line['product_id'],
                        ((int)$line['product_quantity'] * -1), //Pass as negative int (decrement)
                        $line['product_attribute_id']
                    );
                }
                //Add the tax detail to the taxes array
                $taxesArray[$orderDetail->id] = [
                    'id_tax' => HipercalzadoUtils::getIdTaxFromGroup(
                        $orderDetail->id_tax_rules_group,
                        $order->id_address_invoice
                    ),
                    'unit_amount' => round(
                        ($orderDetail->unit_price_tax_incl - $orderDetail->unit_price_tax_excl),
                        4
                    ),
                    'total_amount' => round(
                        ($orderDetail->total_price_tax_incl - $orderDetail->total_price_tax_excl),
                        4
                    )
                ];

                // Guardamos el producto como producto de Hipercalzado
                $WS_order_product = $line['rawItem'];

                $PS_product_id = ($line['product_id'] != HipercalzadoUtils::HIPERCALZADO_ID_NON_EXISTENT_PRODUCT) ? $line['product_id'] : 0;

                if (!$HC_order_product = HipercalzadoOrderProduct::getHipercalzadoOrderProductByPSIdOrderAndHCProductSku($id_hipercalzado_order, $WS_order_product->product_sku)) {
                    $HC_order_product = new HipercalzadoOrderProduct();
                }

                $HC_order_product->id_hipercalzado_order    = (int) $id_hipercalzado_order;
                $HC_order_product->hc_id_product            = (int) $WS_order_product->order_line_id;
                $HC_order_product->st_id_product            = (int) $PS_product_id;
                $HC_order_product->st_id_product_attribute  = (int) $orderDetail->product_attribute_id;
                $HC_order_product->st_id_order_detail       = (int) $orderDetail->id;
                $HC_order_product->hc_product_sku           = pSQL($WS_order_product->product_sku);
                $HC_order_product->quantity                 = (int) $WS_order_product->quantity;
                $HC_order_product->can_refund               = (int) $WS_order_product->can_refund;
                $HC_order_product->date_add                 = pSQL($WS_order_product->created_date);
                $HC_order_product->save();
            }
            // die(dump($orderDetail));
        }

        //insert the tax lines into the db
        foreach ($taxesArray as $lineID => $taxValues) {
            Db::getInstance()->insert(
                'order_detail_tax',
                [
                    'id_order_detail' => pSQL($lineID),
                    'id_tax' => pSQL($taxValues['id_tax']),
                    'unit_amount' => pSQL($taxValues['unit_amount']),
                    'total_amount' => pSQL($taxValues['total_amount'])
                ]
            );
        }
    }

    /**
     * Register's at least one status change
     *
     * @param $order - the order object
     */
    private static function registerOrderStatus($order, $id_order_state, $idEmployee)
    {
        $new_history = new OrderHistory();
        $new_history->id_order = (int) $order->id;
        $new_history->changeIdOrderState((int)$id_order_state, $order, true);

        //Set and save
        $new_history->id_employee       = (int) $idEmployee;
        $new_history->id_order_state    = (int) $id_order_state;
        $new_history->date_add          = date('Y-m-d H:i:s');
        $new_history->date_upd          = date('Y-m-d H:i:s');
        $new_history->save();
    }

    private static function generateInvoice($PS_order)
    {

        // Generamos la factura en caso de no haber sido generada
        $PS_order->setInvoice(true);
        
        //Obtenemos la factura
        $PS_order_invoice = OrderInvoice::getInvoiceByNumber($PS_order->invoice_number);

      //   $tax_calculator =

        $PS_order_invoice->save();
    }

    /**
     * @param $order - The order to register the carrier
     */
    private static function registerOrderCarrier($order, $tracking_number)
    {
        // Get invoice id by nam, if exists
        $invoice = OrderInvoice::getInvoiceByNumber($order->invoice_number);
        $id_order_invoice = Validate::isLoadedObject($invoice) ? $invoice->id : 0;

        $order_carrier                          = new OrderCarrier();
        $order_carrier->id_carrier              = (int) $order->id_carrier;
        $order_carrier->id_order                = (int) $order->id;
        $order_carrier->id_order_invoice        = (int) $id_order_invoice;
        $order_carrier->tracking_number         = pSQL($tracking_number);
        $order_carrier->shipping_cost_tax_excl  = round($order->total_shipping_tax_excl, 6);
        $order_carrier->shipping_cost_tax_incl  = round($order->total_shipping_tax_incl, 6);

        $order_carrier->add();
    }

    public static function getHipercalzadoOrderByOrderId($order_id)
    {

        $id = Db::getInstance()->getValue('SELECT id_hipercalzado_order FROM ' . _DB_PREFIX_ . 'hipercalzado_order WHERE st_id_order = ' . (int)$order_id);

        if (!$id) {
            return false;
        }
        
        $HC_order = new HipercalzadoOrder($id);

        return $HC_order;
    }

    /**
     * Get's the currency object
     *
     * @param $currencyID - The id
     *
     * @return array|null - The currency array
     */
    protected static function getCurrency($currencyID)
    {
        if(is_null($currencyID) || $currencyID == 0 || $currencyID == '') {
            $currencyID = 1;
        }
        $results = Db::getInstance()
            ->ExecuteS(
                '
                SELECT cr.*

                FROM `' . _DB_PREFIX_ . 'currency` cr

                WHERE cr.id_currency = ' . pSQL($currencyID) . '
                '
            );

        return (count($results) > 0 ? $results[0] : null);
    }

    /**
     * This function will return the different payment fragments from the database for a order
     *
     * @param $reference - The Order reference string
     *
     * @return array - The payment fragments array
     */
    public static function getPaymentFragments($reference)
    {
        return Db::getInstance()
            ->ExecuteS(
                '
                SELECT *

                FROM `' . _DB_PREFIX_ . 'order_payment`

                WHERE order_reference = "' . pSQL($reference) . '"
                '
            );
    }
}
